/*-

 * Copyright (c) 2001 Hirokazu WATANABE <gwna@geocities.co.jp>
 *
 *	Version 1.11a 2002/ 1/14
 *
 * THIS IS A TEST RELEASE. IF ALLOCATION WAS FAILED, YOU DEFINES 'DEBUG'
 * WHEN COMPILING YOUR KERNEL AND REPORT TO ME.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF

 *
 *	$FreeBSD: $
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/bus.h>
#include <sys/conf.h>
#include <sys/ioccom.h>
#include <sys/fcntl.h>

#include <machine/bus.h>
#include <machine/resource.h>
#include <sys/rman.h>

#include <isa/isavar.h>

static void sysres_identify (driver_t *, device_t);
static int sysres_probe (device_t);
static int sysres_attach (device_t);
static int sysres_detach (device_t);

#define SYSTEM98_MAX_RESOURCES		(ISA_NPORT)

struct sysres_softc {
	struct resource *ioport_resource[SYSTEM98_MAX_RESOURCES];
	int ioport_resource_id [SYSTEM98_MAX_RESOURCES];
	int ioport_resource_cnt[SYSTEM98_MAX_RESOURCES];

	struct resource *ioport_optional_resource[SYSTEM98_MAX_RESOURCES];
	int ioport_optional_resource_id [SYSTEM98_MAX_RESOURCES];
	int ioport_optional_resource_cnt[SYSTEM98_MAX_RESOURCES];
};

struct system_resource {
	bus_addr_t	addr;
	int	size;
};

/* Very important resources for system works. */
static struct system_resource system98_ioaddr[] = {
	{0x0000, 0x20},	/* DMA, PIC  */
	{0x0020, 0x0a},	/* CAL, DMA */
	{0x0031, 0x07},	/* 8255A */
	{0x0050, 0x03},	/* NMI */
	{0x005c, 0x04},	/* TSTMP */
	{0x00f0, 0x07},	/* CPU */
	{0x0128, 0x01},	/* CAL */
	{0x0439, 0x01},	/* DMA */
	{0x045f, 0x01},	/* TSTMP */
	{0x0534, 0x01},	/* CPU external clock, text-VRAM wait */
	{0x0e05, 0x07},	/* DMA */
	{0x3fd9, 0x07},	/* TCU */
	{0x9892, 0x03},	/* CPU timing */
#ifdef I286
	/* For NPX(80287) communication(Intel RESERVE). */
	/* FreeBSD(98) doesn't correspond on this chip. */
	{0x00f8, 0x06},	/* i286 CPU */
#endif
#ifdef V33A
	/* For V33A internal I/O register for address extention control. */
	{0xff00, 0x80},	/*  */
#endif
#if defined(V50) || defined(V50HL)
	/* For V50, V50HL system I/O area. */
	{0xff00, 0x100},	/*  */
#endif
	{0x0000, 0x00}	/* End of resources mark. */
};

/* Optional resources for system works. */
static struct system_resource system98_optional_ioaddr[] = {
	{0x0041, 0x01},	/* KBD(olpt uses 0x40-0x47) */
	{0x0043, 0x01},	/* KBD(olpt uses 0x40-0x47) */
	{0x0071, 0x01},	/* TCU */
	{0x0073, 0x01},	/* TCU */
	{0x0075, 0x01},	/* TCU */
	{0x0077, 0x01},	/* TCU */
	{0x0091, 0x01},	/* MEM(fdc uses 0x90-0x97) */
	{0x0093, 0x01},	/* MEM(fdc uses 0x90-0x97) */
	{0x09a0, 0x09},	/* DISP */
	{0x0eaa, 0x01},	/* DISP */
	{0x0000, 0x00}	/* End of resources mark. */
};

/* Very important resources for H98 series. */
static struct system_resource system_h98_ioaddr[] = {
	{0x002b, 0x03},	/* DMA */
	{0x0431, 0x01},	/* Power, LED control. */
	{0x9896, 0x01},	/* CPU cache control. */
	{0x98d2, 0x03},	/* PIC */
	{0x0000, 0x00}	/* End of resources mark. */
};

static void sysres_identify(driver_t *drv, device_t parent)
{
	if (device_find_child(parent, "sysres", 0) == NULL)
		if (BUS_ADD_CHILD(parent, 100, "sysres", 0) == NULL)
			device_printf(parent, "sysres attach failed.\n");
}

static int sysres_probe(device_t dev)
{
	int iobase = bus_get_resource_start(dev, SYS_RES_IOPORT, 0);

	if (iobase) {	/* No offset needed for this device. */
		return ENXIO;
	}
	device_set_desc(dev, "System resource");

	return 0;
}

static int sysres_attach(device_t dev)
{
	struct sysres_softc *sc = device_get_softc(dev);
	struct resource *res;
	int rid, i;

	rid = 0;

	/* Allocate system resource. */
	if (bootverbose)
		printf("sysres: allocate system resources");
	for (i = 0; system98_ioaddr[i].size; i ++) {
		sc->ioport_resource_id[i] = -1;
		sc->ioport_resource_cnt[i] = 0;
		res = bus_alloc_resource(dev, SYS_RES_IOPORT, &rid,
					 system98_ioaddr[i].addr,
					 system98_ioaddr[i].addr +
					 system98_ioaddr[i].size - 1,
					 system98_ioaddr[i].size,
					 RF_ACTIVE);
		if (!res) {
#ifndef DEBUG	/* If 'DEBUG' was defined, no check for allocate resources. */
			if (bootverbose)
				printf(" failed.\n");
			return (ENOENT);
#else
			if (bootverbose)
				printf("*[%x:%x]",
					system98_ioaddr[i].addr,
					system98_ioaddr[i].size);
#endif
		}
		else {
			if (bootverbose)
				printf(".");
		}
		sc->ioport_resource_id[i] = rid;
		sc->ioport_resource[i] = res;
		sc->ioport_resource_cnt[i] = system98_ioaddr[i].size;
		rid ++;
	}
	if (bootverbose)
		printf("done[%d].\n",rid);

	/* Allocate optional system resource. */
	/* No errors occurs if it failed to allocate. */
	if (bootverbose)
		printf("sysres: allocate optional system resources");
	for (i = 0; system98_optional_ioaddr[i].size; i ++) {
		sc->ioport_optional_resource_id[i] = -1;
		sc->ioport_optional_resource_cnt[i] = 0;
		res = bus_alloc_resource(dev, SYS_RES_IOPORT, &rid,
					 system98_optional_ioaddr[i].addr,
					 system98_optional_ioaddr[i].addr +
					 system98_optional_ioaddr[i].size - 1,
					 system98_optional_ioaddr[i].size,
					 RF_ACTIVE);
		if (!res) {
			if (bootverbose)
				printf("*");
		}
		else {
			if (bootverbose)
				printf(".");
		}
		sc->ioport_optional_resource_id[i] = rid;
		sc->ioport_optional_resource[i] = res;
		sc->ioport_optional_resource_cnt[i] = system98_ioaddr[i].size;
		rid ++;
	}
	if (bootverbose)
		printf("done[%d].\n",rid);

	return 0;
}

static int sysres_detach(device_t dev)
{
	/* This function is not supported. */
	return EBUSY;
}

static device_method_t sysres_methods[] = {
	DEVMETHOD(device_identify,	sysres_identify),
	DEVMETHOD(device_probe,		sysres_probe),
	DEVMETHOD(device_attach,	sysres_attach),
	DEVMETHOD(device_detach,	sysres_detach),
	{0, 0}
};

static driver_t sysres_driver = {
	"sysres",
	sysres_methods,
	sizeof(struct sysres_softc),
};

devclass_t sysres_devclass;
DRIVER_MODULE(sysres, isa, sysres_driver, sysres_devclass, 0, 0);
