/*-
 * Copyright (c) 1999 Brad Forschinger
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $FreeBSD: src/sys/modules/syscons/fire/fire_saver.c,v 1.6.2.1 2000/05/10 14:01:23 obrien Exp $
 */

/*
 * brad forschinger, 19990504 <retch@flag.blackened.net>
 * 
 * written with much help from warp_saver.c
 * 
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/module.h>
#include <sys/syslog.h>
#include <sys/consio.h>
#include <sys/malloc.h>
#include <sys/fbio.h>
#include <sys/random.h>

#include <dev/fb/fbreg.h>
#include <dev/fb/splashreg.h>
#include <dev/syscons/syscons.h>

static int      blanked;
static u_char   fire_pal[768];
static u_char  *buf;
static u_char  *vid;
static int banksize, scrmode, bpsl, scrw, scrh;

static int
fire_saver(video_adapter_t *adp, int blank)
{
    int             x, y;
    int             o, p, pl;

    if (blank) {
	if (blanked <= 0) {
	    int             red, green, blue;
	    int             palette_index;

	    /* build and load palette */
	    red = green = blue = 0;
	    for (palette_index = 0; palette_index < 256; palette_index++) {
		red++;
		if (red > 128)
		    green += 2;

		fire_pal[(palette_index * 3) + 0] = red;
		fire_pal[(palette_index * 3) + 1] = green;
		fire_pal[(palette_index * 3) + 2] = blue;
	    }
	    pl = splhigh();
	    set_video_mode(adp, scrmode);
	    load_palette(adp, fire_pal);
	    blanked++;
	    vid = (u_char *)adp->va_window;
	    banksize = adp->va_window_size;
	    bpsl = adp->va_line_width;
	    splx(pl);
	}

	if (buf == NULL) {
	    buf = (u_char *)malloc(scrw * (scrh + 1), M_DEVBUF, M_NOWAIT);
	    bzero(buf, scrw * (scrh + 1));
	}
	if (buf == NULL) {
	    return 0;
	}
	/* make a new bottom line */
	for (x = 0, y = scrh; x < scrw; x++)
	    buf[x + (y * bpsl)] = random() % 160 + 96;

	/* fade the flames out */
	for (y = 0; y < scrh; y++) {
	    for (x = 0; x < scrw; x++) {
		buf[x + (y * scrw)] = (buf[(x + 0) + ((y + 0) * scrw)] +
					 buf[(x - 1) + ((y + 1) * scrw)] +
					 buf[(x + 0) + ((y + 1) * scrw)] +
				     buf[(x + 1) + ((y + 1) * scrw)]) / 4;
		if (buf[x + (y * scrw)] > 0)
		    buf[x + (y * scrw)]--;
	    }
	}

	/* blit our buffer into video ram */
	for (y = 0, p = 0, o = 0; y < scrh; y++, p += bpsl) {
	    while (p > banksize) {
		p -= banksize;
		o += banksize;
	    }
	    set_origin(adp, o);
	    if (p + scrw < banksize) {
		bcopy(buf + y * scrw, vid + p, scrw);
	    } else {
		bcopy(buf + y * scrw, vid + p, banksize - p);
		set_origin(adp, (o += banksize));
		bcopy(buf + y * scrw + (banksize - p), vid,
		      scrw - (banksize - p));
		p -= banksize;
	    }
	}
    } else {
	blanked = 0;
    }

    return 0;
}

static int
fire_initialise(video_adapter_t *adp)
{
    video_info_t    info;

    if (!get_mode_info(adp, M_VGA_CG320, &info)) {
	scrmode = M_VGA_CG320;
    } else if (!get_mode_info(adp, M_PC98_PEGC640x480, &info)) {
	scrmode = M_PC98_PEGC640x480;
    } else if (!get_mode_info(adp, M_PC98_PEGC640x400, &info)) {
	scrmode = M_PC98_PEGC640x400;
    } else {
        log(LOG_NOTICE, "logo_saver: no suitable graphics mode\n");
	return ENODEV;
    }
    
    scrw = info.vi_width;
    scrh = info.vi_height;
    blanked = 0;
    buf = NULL;

    return 0;
}

static int
fire_terminate(video_adapter_t *adp)
{
    if (buf != NULL) {
	free(buf, M_DEVBUF);
	buf = NULL;
    }
    return 0;
}

static scrn_saver_t fire_module = {
    "fire_saver", fire_initialise, fire_terminate, fire_saver, NULL
};

SAVER_MODULE(fire_saver, fire_module);
