/*-
 * Copyright (c) 2001,2002 IMAI Takeshi
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $FreeBSD$
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/disk.h>
#include <sys/module.h>
#include <sys/bus.h>
#include <sys/malloc.h>
#include <sys/devicestat.h>
#include <sys/sysctl.h>
#include <machine/stdarg.h>
#include <machine/resource.h>
#include <machine/bus.h>
#include <machine/clock.h>
#include <sys/rman.h>
#include <isa/isavar.h>
#include <pc98/pc98/pc98_machdep.h>
#include <dev/ata/ata-all.h>
#include <dev/ata/atapi-all.h>
#include "ata_if.h"

/* global vars */
devclass_t nac_devclass;

/* local vars */
static bus_addr_t nac_ports[] = {
    0x0, 0x2, 0x4, 0x6, 0x8, 0xa, 0xc, 0xe
};

struct nac_softc {
    struct resource *io;
    struct resource *altio;
    struct resource *irq;
    struct resource *bankio;
    int device[2];
    int lock;
};

/*
 * Vaild bank check. (return devices)
 * Some PC98x1 seems to have bank #1, but cannot be used.
 * There are illegal image of bank #0.
 */
static int
nac_check_bank(device_t dev, int bank)
{
    struct nac_softc* scp = device_get_softc(dev);
    int timeout, devices, mask = 0;
    u_int8_t a, b, ostat0, ostat1;
    u_int8_t status0 = ATA_S_BUSY, status1 = ATA_S_BUSY;
    int s = splbio();

    /* select bank */
    bus_space_write_1(rman_get_bustag(scp->bankio),
		      rman_get_bushandle(scp->bankio), 0, bank);

    /* do we have any signs of ATA/ATAPI HW being present ? */
    /* get the current status of the devices */
    ATA_OUTB(scp->io, ATA_DRIVE, ATA_D_IBM | ATA_SLAVE);
    DELAY(10);
    ostat1 = ATA_INB(scp->io, ATA_STATUS);
    ATA_OUTB(scp->io, ATA_DRIVE, ATA_D_IBM | ATA_MASTER);
    DELAY(10);
    ostat0 = ATA_INB(scp->io, ATA_STATUS);

    if ((ostat0 & 0xf8) != 0xf8 && ostat0 != 0xa5)
	mask |= 0x01;
    if ((ostat1 & 0xf8) != 0xf8 && ostat1 != 0xa5)
	mask |= 0x02;

    if (bootverbose)
	printf("nac%d: bank%d - mask=%02x ostat0=%02x ostat1=%02x\n",
		device_get_unit(dev), bank, mask, ostat0, ostat1);

    /* reset channel */
    ATA_OUTB(scp->altio, 0, ATA_A_IDS | ATA_A_RESET);
    DELAY(10000);
    ATA_OUTB(scp->altio, 0, ATA_A_IDS);
    DELAY(100000);
    ATA_INB(scp->io, ATA_ERROR);
    devices = 0;

    /* wait for BUSY to go inactive */
    for (timeout = 0; timeout < 310000; timeout++) {
	if (status0 & ATA_S_BUSY) {
	    ATA_OUTB(scp->io, ATA_DRIVE, ATA_D_IBM | ATA_MASTER);
	    DELAY(10);
	    status0 = ATA_INB(scp->io, ATA_STATUS);
	    if (!(status0 & ATA_S_BUSY)) {
		/* check for ATAPI signature while its still there */
		a = ATA_INB(scp->io, ATA_CYL_LSB);
		b = ATA_INB(scp->io, ATA_CYL_MSB);
		if (bootverbose)
		    printf("nac%d: bank%d-master - ATAPI probe a=%02x b=%02x\n",
			    device_get_unit(dev), bank, a, b);
		if (a == ATAPI_MAGIC_LSB && b == ATAPI_MAGIC_MSB)
		    devices |= ATA_ATAPI_MASTER;
	    }
	}
	if (status1 & ATA_S_BUSY) {
	    ATA_OUTB(scp->io, ATA_DRIVE, ATA_D_IBM | ATA_SLAVE);
	    DELAY(10);
	    status1 = ATA_INB(scp->io, ATA_STATUS);
	    if (!(status1 & ATA_S_BUSY)) {
		/* check for ATAPI signature while its still there */
		a = ATA_INB(scp->io, ATA_CYL_LSB);
		b = ATA_INB(scp->io, ATA_CYL_MSB);
		if (bootverbose)
		    printf("nac%d: bank%d-slave - ATAPI probe a=%02x b=%02x\n",
			    device_get_unit(dev), bank, a, b);
		if (a == ATAPI_MAGIC_LSB && b == ATAPI_MAGIC_MSB)
		    devices |= ATA_ATAPI_SLAVE;
	    }
	}
	if (!mask)		/* no device */
	    break;
	if (mask == 0x01)	/* wait for master only */
	    if (!(status0 & ATA_S_BUSY))
		break;
	if (mask == 0x02)	/* wait for slave only */
	    if (!(status1 & ATA_S_BUSY))
		break;
	if (mask == 0x03)	/* wait for both master & slave */
	    if (!(status0 & ATA_S_BUSY) && !(status1 & ATA_S_BUSY))
		break;
	DELAY(100);
    }
    DELAY(10);
    ATA_OUTB(scp->altio, 0, ATA_A_4BIT);

    if (status0 & ATA_S_BUSY)
	mask &= ~0x01;
    if (status1 & ATA_S_BUSY)
	mask &= ~0x02;
    if (bootverbose)
	printf("nac%d: bank%d - mask=%02x status0=%02x status1=%02x\n",
		device_get_unit(dev), bank, mask, status0, status1);

    if (!mask) {
	if (bootverbose)
	    printf("nac%d: bank%d - no device.\n",
		    device_get_unit(dev), bank);
	goto done_check;
    }

    status0 = 0;
    if (mask & 0x01 && ostat0 != 0x00 && !(devices & ATA_ATAPI_MASTER)) {
	ATA_OUTB(scp->io, ATA_DRIVE, ATA_D_IBM | ATA_MASTER);
	DELAY(10);
	ATA_OUTB(scp->io, ATA_ERROR, 0x58);
	ATA_OUTB(scp->io, ATA_CYL_LSB, 0xa5);
	a = ATA_INB(scp->io, ATA_ERROR);
	b = ATA_INB(scp->io, ATA_CYL_LSB);
	if (bootverbose)
	    printf("nac%d: bank%d-master - ATA probe a=%02x b=%02x\n",
		    device_get_unit(dev), bank, a, b);
	if (a != 0x58 && b == 0xa5)
	    devices |= ATA_ATA_MASTER;
	status0 = ATA_INB(scp->io, ATA_STATUS);
    }
    status1 = 0;
    if (mask & 0x02 && ostat1 != 0x00 && !(devices & ATA_ATAPI_SLAVE)) {
	ATA_OUTB(scp->io, ATA_DRIVE, ATA_D_IBM | ATA_SLAVE);
	DELAY(10);
	ATA_OUTB(scp->io, ATA_ERROR, 0x58);
	ATA_OUTB(scp->io, ATA_CYL_LSB, 0xa5);
	a = ATA_INB(scp->io, ATA_ERROR);
	b = ATA_INB(scp->io, ATA_CYL_LSB);
	if (bootverbose)
	    printf("nac%d: bank%d-slave - ATA probe a=%02x b=%02x\n",
		    device_get_unit(dev), bank, a, b);
	if (a != 0x58 && b == 0xa5)
	    devices |= ATA_ATA_SLAVE;
	status1 = ATA_INB(scp->io, ATA_STATUS);
    }
    if (bootverbose)
	printf("nac%d: bank%d - devices=%02x status0=%02x status1=%02x\n",
		device_get_unit(dev), bank, devices, status0, status1);

    /* clear any interrupts ? */
    DELAY(100);
    ATA_INB(scp->io, ATA_STATUS);

done_check:
    splx(s);
    return devices;
}

static int
nac_probe(device_t dev)
{
    int rid;
    struct resource *port;
    u_long tmp;

    if (isa_get_port(dev) != ATA_PC98_PRIMARY)
	return ENOMEM;

    /* allocate the port range */
    rid = ATA_IOADDR_RID;
    port = isa_alloc_resourcev(dev, SYS_RES_IOPORT, &rid, nac_ports,
			       ATA_IOSIZE, RF_ACTIVE);
    if (!port)
       return ENOMEM;
    isa_load_resourcev(port, nac_ports, ATA_IOSIZE);

    /* allocate the altport range */
    rid = ATA_IOADDR_RID + ATA_IOSIZE + 1;
    if (bus_get_resource(dev, SYS_RES_IOPORT, rid, &tmp, &tmp)) {
	bus_set_resource(dev, SYS_RES_IOPORT, rid,
			 rman_get_start(port) + ATA_PC98_ALTOFFSET,
			 ATA_ALTIOSIZE);
    }

    /* allocate the bank port range */
    rid = ATA_IOADDR_RID + ATA_IOSIZE + ATA_ALTIOSIZE + 1;
    if (bus_get_resource(dev, SYS_RES_IOPORT, rid, &tmp, &tmp)) {
	bus_set_resource(dev, SYS_RES_IOPORT, rid,
			 ATA_PC98_BANK, ATA_PC98_BANKSIZE);
    }

    bus_release_resource(dev, SYS_RES_IOPORT, 0, port);
    device_set_desc(dev, "NEC PC-98x1 ATA Controller");
    return 0;
}

static int
nac_attach(device_t dev)
{
    struct nac_softc *scp = device_get_softc(dev);
    int rid;
    device_t child;
    int unit, flags, pc98_bios;

    if (resource_int_value("nac", 0, "flags", &flags) != 0)
	flags = 0;

    /* clear lock */
    scp->lock = 0;

    /* allocate resources */
    rid = ATA_IOADDR_RID;
    scp->io = isa_alloc_resourcev(dev, SYS_RES_IOPORT, &rid, nac_ports,
				  ATA_IOSIZE, RF_ACTIVE | RF_SHAREABLE);
    if (!scp->io)
       return ENOMEM;
    isa_load_resourcev(scp->io, nac_ports, ATA_IOSIZE);

    rid = ATA_IOADDR_RID + ATA_IOSIZE + 1;
    scp->altio = bus_alloc_resource(dev, SYS_RES_IOPORT, &rid,
				    rman_get_start(scp->io)+ATA_PC98_ALTOFFSET,
				    ~0, ATA_ALTIOSIZE, RF_ACTIVE|RF_SHAREABLE);
    if (!scp->altio)
	return ENOMEM;

    rid = ATA_IOADDR_RID + ATA_IOSIZE + ATA_ALTIOSIZE + 1;
    scp->bankio = bus_alloc_resource(dev, SYS_RES_IOPORT, &rid,
				     ATA_PC98_BANK, ~0,
				     ATA_PC98_BANKSIZE, RF_ACTIVE);
    if (!scp->bankio)
	return ENOMEM;

    rid = 0;
    scp->irq = bus_alloc_resource(dev, SYS_RES_IRQ, &rid,
				  0, ~0, 1, RF_ACTIVE | RF_SHAREABLE);

    /* check presence of secondary channel */
    if (flags & 0x80) {
	scp->device[0] = -1;
	scp->device[1] = -1;
    }
    else {
	/* bank check */
	scp->device[0] = nac_check_bank(dev, 0);
	scp->device[1] = nac_check_bank(dev, 1);
	if ((scp->device[0] == scp->device[1]) &&
	    (scp->device[1] & ATA_ATA_MASTER)) {
	    pc98_bios = PC98_SYSTEM_PARAMETER(0x457);
	    if (bootverbose)
		printf("nac%d: ATA MASTER device on both bank (bios parameter %x)\n",
		       device_get_unit(dev), pc98_bios);
	    if (!(pc98_bios & 0x40)) {
		scp->device[1] = 0;
		printf("nac%d: found invalid bank 1\n", device_get_unit(dev));
	    }
	}

    }

    for(unit=0; unit<2; unit++) {
	if (!(child = device_add_child(dev, "ata", unit)))
	    return ENOMEM;
	device_set_ivars(child, (void *)unit);
    }

    return bus_generic_attach(dev);
}

static struct resource *
nac_alloc_resource(device_t dev, device_t child, int type, int *rid,
		   u_long start, u_long end, u_long count, u_int flags)
{
    struct nac_softc *scp = device_get_softc(dev);

    if (type == SYS_RES_IOPORT) {
	switch (*rid) {
	case ATA_IOADDR_RID:
	    return scp->io;
	case ATA_ALTADDR_RID:
	    return scp->altio;
	default:
	    return 0;
	}
    }

    if (type == SYS_RES_IRQ) {
	return scp->irq;
    }
    return 0;
}

static int
nac_release_resource(device_t dev, device_t child, int type, int rid,
                    struct resource *r)
{
    return 0; 
}

static int
nac_setup_intr(device_t dev, device_t child, struct resource *irq,
               int flags, driver_intr_t *intr, void *arg,
               void **cookiep)
{
    return BUS_SETUP_INTR(device_get_parent(dev), dev, irq,
			  flags, intr, arg, cookiep);
}

static int
nac_print_child(device_t dev, device_t child)
{
    struct ata_channel *ch = device_get_softc(child);
    int retval = 0;

    retval += bus_print_child_header(dev, child);
    retval += printf(" at bank %d", ch->unit);
    retval += bus_print_child_footer(dev, child);
    return retval;
}

static int
nac_select_bank(device_t dev, int bank)
{
    struct nac_softc *scp = device_get_softc(dev);

    bus_space_write_1(rman_get_bustag(scp->bankio),
		      rman_get_bushandle(scp->bankio), 0, bank);
    return 0;
}

static int
nac_lock_controller(device_t dev)
{
    struct nac_softc *scp = device_get_softc(dev);
    if (scp->lock) {
#if 0
printf("nac_lock_controller: lock failed.\n");
#endif
	return 0;
    }
    scp->lock = 1;
    return 1;
}

static void
nac_unlock_controller(device_t dev, int value)
{
    struct nac_softc *scp = device_get_softc(dev);
    scp->lock = 0;
}

static device_method_t nac_methods[] = {
    /* device_interface */
    DEVMETHOD(device_probe,	nac_probe),
    DEVMETHOD(device_attach,	nac_attach),
    DEVMETHOD(device_resume,	bus_generic_resume),

    /* bus methods */
    DEVMETHOD(bus_alloc_resource,	nac_alloc_resource),
    DEVMETHOD(bus_release_resource,	nac_release_resource),
    DEVMETHOD(bus_setup_intr,		nac_setup_intr),
    DEVMETHOD(bus_print_child,		nac_print_child),

    /* ata methods */
    DEVMETHOD(ata_select_bank,		nac_select_bank),
    DEVMETHOD(ata_lock_controller,	nac_lock_controller),
    DEVMETHOD(ata_unlock_controller,	nac_unlock_controller),
    { 0, 0 }
};

static driver_t nac_driver = {
    "nac",
    nac_methods,
    sizeof(struct nac_softc),
};

DRIVER_MODULE(nac, isa, nac_driver, nac_devclass, 0, 0);
